package easik.sketch.util.ini;

import java.io.File;
import java.io.RandomAccessFile;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;

/**
 * Class to read in and write out ini files
 * 
 * @author Ian Rutherford 2001
 * @author Kevin Green 2006
 * @since 2006-07-11 Kevin Green
 * @version 2006-08-04 Kevin Green
 */
public class IniFile
{
	/**
	 * Hash map of the IniFileNode, indexed by the name of the header
	 */
	private HashMap<String, IniFileNode> _nodey;

	/**
	 * Default constructor
	 */
	public IniFile()
	{
		_nodey = new HashMap<String, IniFileNode>();
	}

	/**
	 * Adds a new IniFileNode for the new header
	 * 
	 * @param name The title of the new header
	 * @return True if the addition worked, false otherwise
	 */
	public boolean addHeader(String name)
	{
		int i;

		// check for bad input
		if (name.length() == 0)
			return false;

		// make sure the name is alphanumeric
		for (i=0; i<name.length(); i++)
		{
			if (!((name.charAt(i) >= 'a' && name.charAt(i) <= 'z') || (name.charAt(i) >= 'A' && name.charAt(i) <= 'Z') || (name.charAt(i) >= '0' && name.charAt(i) <= '9') || name.charAt(i) == ' '))
				return false;
		}			

		// now add the header to the list
		//make sure it doesn't already exist
		if(_nodey.containsKey(name)){
			return false;
		}
		else{
			_nodey.put(name, new IniFileNode(name));
		}
		
		return true;
	}

	/**
	 * Removes the given header
	 * 
	 * @param name The name of the header to be removed
	 * @return True if it was removed properly, false otherwise
	 */
	public boolean removeHeader(String name)
	{
		if(!_nodey.containsKey(name)){
			return false;
		}
		else{
			_nodey.remove(name);
			return true;
		}
	}

	/**
	 * Removes all headers
	 */
	public void eraseAll()
	{
		_nodey.clear();
	}

	/**
	 * Return the IniFileNode for the given header
	 * @param name The name of the header
	 * @return The IniFileNode for the given header, or null if the header does not exist
	 */
	public IniFileNode getHeader(String name)
	{
		if(!_nodey.containsKey(name)){
			return null;
		}
		else{
			return _nodey.get(name);
		}
	}
	
	/**
	 * Writes the ini to a file
	 * 
	 * @param filename The name of the ini file
	 * @return True if written properly, false otherwise
	 */
	public boolean writeIni(String filename)
	{
		RandomAccessFile file;

		//Delete the old file.
		File oldFile = new File(filename);
		oldFile.delete();
		
		// try to open the file
		try
		{
			file = new RandomAccessFile(filename, "rw");
		}
		catch (IOException e)
		{
			// if we can't open the file, return false
			return false;
		}

		String data = getData();

		// try to write all the data to the file in one shot
		try
		{
			file.write(data.getBytes());
			file.close();
		}
		catch (IOException e)
		{
			// if we can't write to the file, return false
			return false;
		}
		// operation successful; return true
		return true;
	}

	/**
	 * Reads the ini from a file
	 * 
	 * @param filename The name of the ini file
	 * @return True if read in properly, false otherwise
	 */
	public boolean readIni(String filename)
	{
		String temp;
		String curHeader = "";
		int j = 0, m;
		RandomAccessFile file;
		// erase all data in memory before loading the new file
		eraseAll();

		// try to open the file for input
		try
		{
			file = new RandomAccessFile(filename, "r");
			while (true)
			{
				// try reading a line from the file
				try
				{
					temp = file.readLine();
				}
				catch (IOException e)
				{
					// if there is an error reading any more data, then we're done
					file.close();
					return false;
				}
				// if we're at the end of the file (when temp is null), then we're done
				if (temp == null){
					file.close();
					return true;
				}

				// if the line is blank, do nothing. 
				if (temp.length() == 1 || temp.length() == 0){
					//Do nothing
				}
				// if the line is a header surrounded by square brackets, then add it to the list
				else if (temp.charAt(0) == '[' && temp.charAt(temp.length()-1) == ']')
				{
					curHeader = temp.substring(1, temp.length()-1);
					addHeader(temp.substring(1, temp.length()-1));
				}

				// if there is one square bracket but not another, then the line is invalid and
				// therefore the file itself is an invalid .ini file
				else if (temp.charAt(0) == '[')
				{
					eraseAll();
					file.close();
					return false;
				}

				// otherwise, check to see if the line is an item to be added to the list
				else
				{
					// find the equals character ('='), at position j
					j = 0;
					while (j < temp.length() && temp.charAt(j) != '=')
						j++;
					// if the first or last character is a '=' then the line is invalid
					if (j == 0 || j >= temp.length())
					{
						eraseAll();
						file.close();
						return false;
					}
					m = j;

					// the next section removes whitespace between the '=' and the data
					j++;
					m--;

					// find the beginning of the text to the right of the '='
					while (j < temp.length() && temp.charAt(j) == ' ')
						j++;

					// find the end of the text to the left of the '='
					while (m >= 0 && temp.charAt(m) == ' ')
						m--;

					// if the left side is all whitespace, then the line is invalid
					if (m < 0)
					{
						eraseAll();
						file.close();
						return false;
					}

					String rhs = temp.substring(j, temp.length());
					//Remove carriage returns and line feeds from the ini data
					for (int i=0; i < rhs.length(); i++)
					{
						if ((rhs.charAt(i) == '\n') || (rhs.charAt(i) == '\r'))
						{
							if (i+1 >= rhs.length())
								rhs = rhs.substring(0, i);
							else
								rhs = rhs.substring(0, i) + rhs.substring(i+1);
						}
					}

					// now we have a valid item, so add it to the list
					String lhs = temp.substring(0,m+1);
					getHeader(curHeader).addItem(lhs, rhs);
				}
			}
		}
		catch (IOException e)
		{
			// if there is an error opening the file, return false
			return false;
		}
	}

	/**
	 * Returns a string containing the formatted .ini file data
	 * 
	 * @return A string containing the formatted .ini file data
	 */
	public String getData()
	{
		String data = "";
		Iterator nodeIt = _nodey.keySet().iterator();
		while(nodeIt.hasNext()){
			IniFileNode curNode = _nodey.get(nodeIt.next());
			if(curNode.numItems() != 0){
				data += "[" + curNode.getTitle() + "]\n";
				Iterator itemIt = curNode.getAllItems().keySet().iterator();
				while(itemIt.hasNext()){
					String curItem = (String)itemIt.next();
					data += curItem + " = " + curNode.getItem(curItem) + "\n";
				}
				data += "\n";
			}
		}
		return data;
	}
}
